
#include "Nsf_Emu.h"

#if !NSF_EMU_APU_ONLY
	#include "Nes_Namco_Apu.h"
#endif

#include "blargg_source.h"

int Nsf_Emu::cpu_read( nes_addr_t addr )
{
	int result;
	
	result = cpu::low_mem [addr & 0x7FF];
	if ( !(addr & 0xE000) )
		goto exit;
	
	result = *cpu::get_code( addr );
	if ( addr > 0x7FFF )
		goto exit;
	
	result = sram [addr & (sizeof sram - 1)];
	if ( addr > 0x5FFF )
		goto exit;
	
	if ( addr == Nes_Apu::status_addr )
		return apu.read_status( cpu::time() );
	
	#if !NSF_EMU_APU_ONLY
		if ( addr == Nes_Namco_Apu::data_reg_addr && namco )
			return namco->read_data();
	#endif
	
	result = addr >> 8; // simulate open bus
	
	if ( addr != 0x2002 )
		dprintf( "Read unmapped $%.4X\n", (unsigned) addr );
	
exit:
	return result;
}

void Nsf_Emu::cpu_write( nes_addr_t addr, int data )
{
	// unsigned converts negative into large positive value
	{
		unsigned sram_addr = addr - 0x6000;
		if ( sram_addr < sizeof sram )
		{
			sram [sram_addr] = data;
			return;
		}
	}
	{
		int temp = addr & 0x7FF;
		if ( !(addr & 0xE000) )
		{
			cpu::low_mem [temp] = data;
			return;
		}
	}
	
	if ( unsigned (addr - Nes_Apu::start_addr) <= Nes_Apu::end_addr - Nes_Apu::start_addr )
	{
		apu.write_register( cpu::time(), addr, data );
		return;
	}
	
	unsigned bank = addr - bank_select_addr;
	if ( bank < bank_count )
	{
		if ( data < total_banks )
			cpu::map_code( (bank + 8) * page_size, page_size, &rom [data * page_size] );
		else
			dprintf( "Bank %d out of range (%d banks total)\n", data, (int) total_banks );
		return;
	}
	
	cpu_write_misc( addr, data );
}

#define CPU_READ( cpu, addr, time )         STATIC_CAST(Nes_Emu&,*cpu).cpu_read( addr )
#define CPU_WRITE( cpu, addr, data, time )  STATIC_CAST(Nes_Emu&,*cpu).cpu_write( addr, data )
